<?php
/**
 * Elementor Widget: Ultra – Category Grid (Layout 2)
 * -------------------------------------------------
 * نمایش دسته‌بندی محصولات ووکامرس به‌صورت شبکه‌ای
 * @author  Ultra Market
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

class Elementor_ultra_category_grid_widget extends Widget_Base {

    /* ───────── Basics ───────── */

    public function get_name()        { return 'ultra_category_grid'; }
    public function get_title()       { return 'دسته‌بندی (شبکه‌ای)'; }
    public function get_icon()        { return 'eicon-product-categories'; }
    public function get_categories()  { return [ 'ultra' ]; }
    public function get_keywords()    { return [ 'product', 'category', 'grid' ]; }

    /* ───────── Controls ───────── */

    protected function register_controls() {

        $this->start_controls_section(
            'content_section',
            [
                'label' => __( 'محتوا', 'ultra-market' ),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        // انتخاب دسته‌بندی‌ها
        $this->add_control(
            'selected_categories',
            [
                'label'       => 'دسته‌بندی‌ ها',
                'type'        => Controls_Manager::SELECT2,
                'multiple'    => true,
                'label_block' => true,
                'options'     => $this->get_cat_options(),
                'description' => 'اگر انتخاب نکنید، همهٔ دسته‌ها نمایش داده می‌شود.',
            ]
        );

        $this->add_control(
            'open_in_new_tab',
            [
                'label'        => 'باز شدن در تب جدید',
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'label_on'     => 'بله',
                'label_off'    => 'خیر',
                'return_value' => 'yes',
                'default'      => '',
            ]
        );


        // نمایش دسته‌های خالی؟
        $this->add_control(
            'show_empty',
            [
                'label'        => 'نمایش دسته‌های خالی',
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => 'بله',
                'label_off'    => 'خیر',
                'return_value' => 'yes',
                'default'      => '',
            ]
        );

        $this->add_control(
            'exclude_uncategorized',
            [
                'label'        => 'حذف دسته‌بندی پیش‌فرض (Uncategorized)',
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'label_on'     => 'بله',
                'label_off'    => 'خیر',
                'return_value' => 'yes',
                'default'      => 'yes',
            ]
        );


        // کارت «سایر»
        $this->add_control(
            'show_other_card',
            [
                'label'        => 'نمایش کارت «سایر»',
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => 'نمایش',
                'label_off'    => 'مخفی',
                'return_value' => 'yes',
                'default'      => 'yes',
            ]
        );

        $this->add_control(
            'other_card_link',
            [
                'label'       => 'لینک کارت «سایر»',
                'type'        => Controls_Manager::URL,
                'default'     => [
                    'url'         => get_permalink( wc_get_page_id( 'shop' ) ),
                    'is_external' => false,
                ],
                'condition'   => [ 'show_other_card' => 'yes' ],
                'label_block' => true,
            ]
        );

        $this->end_controls_section();

        /* ───────── Style ───────── */

        $this->start_controls_section(
            'style_section',
            [
                'label' => __( 'استایل', 'ultra-market' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'bg_color',
            [
                'label'     => 'رنگ پس‌زمینه ',
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category-item' => 'background: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'bg_hover_color',
            [
                'label'     => 'رنگ پس‌زمینه هاور',
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category-item:hover' => 'background: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => 'رنگ عنوان',
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category-item h5' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /* ───────── Render ───────── */

    protected function render() {

        $settings          = $this->get_settings_for_display();
        $selected_cats     = $settings['selected_categories'] ?: [];
        $show_empty        = ! empty( $settings['show_empty'] );
        $show_other_card   = ! empty( $settings['show_other_card'] );
        $other_card_link   = $settings['other_card_link']['url'] ?? '#';

        $args = [
            'taxonomy'   => 'product_cat',
            'orderby'    => 'name',
            'order'      => 'ASC',
            'hide_empty' => ! $show_empty,
        ];
        if ( $selected_cats ) {
            $args['include'] = $selected_cats;
        }

        $cats = get_terms( $args );

        if ( is_wp_error( $cats ) || empty( $cats ) ) {
            echo '<p>'. esc_html__( 'دسته‌ای یافت نشد.', 'ultra-market' ) .'</p>';
            return;
        }
        if ( ! is_wp_error( $cats ) && $settings['exclude_uncategorized'] === 'yes' ) {

            $default_cat_id = get_option( 'default_product_cat' ); // دسته پیش‌فرض ووکامرس
            if ( ! $default_cat_id ) {
                $default_cat_id = get_option( 'default_category' ); // دسته پیش‌فرض پست‌ها (fallback)
            }

            // حذف هم بر اساس ID و هم برای اسلاگ‌های معروف فارسی و انگلیسی
            $cats = array_filter( $cats, function( $cat ) use ( $default_cat_id ) {
                $uncategorized_slugs = [ 'uncategorized', 'دسته-بندی-نشده' ];
                return $cat->term_id != $default_cat_id && ! in_array( $cat->slug, $uncategorized_slugs );
            } );
        }
        $link_target = ! empty( $settings['open_in_new_tab'] ) && $settings['open_in_new_tab'] === 'yes' ? ' target="_blank"' : '';
        ?>
        <section class="ultra-category-grid">
            <?php
            foreach ( $cats as $cat ) :

                $thumb_id  = get_term_meta( $cat->term_id, 'thumbnail_id', true );
                $thumb_url = $thumb_id ? wp_get_attachment_url( $thumb_id ) : wc_placeholder_img_src();
                ?>
            <a class="category-item" href="<?php echo esc_url( get_term_link( $cat ) ); ?>"<?php echo $link_target; ?>>
                    <img src="<?php echo esc_url( $thumb_url ); ?>"
                         alt="<?php echo esc_attr( $cat->name ); ?>" />
                    <h5><?php echo esc_html( $cat->name ); ?></h5>
                </a>
            <?php
            endforeach;

            /* کارت «سایر» */
            if ( $show_other_card ) : ?>
                <a class="category-item category-item--other" href="<?php echo esc_url( $other_card_link ); ?>">
					<span class="dots">
						<ul><li></li><li></li><li></li></ul>
					</span>
                    <h5><?php _e( 'سایر', 'ultra-market' ); ?></h5>
                </a>
            <?php endif; ?>
        </section>
        <?php
    }

    /* ───────── Helpers ───────── */

    private function get_cat_options() {

        $terms = get_terms( [
            'taxonomy'   => 'product_cat',
            'orderby'    => 'name',
            'order'      => 'ASC',
            'hide_empty' => false,
        ] );

        $options = [];
        if ( ! is_wp_error( $terms ) ) {
            foreach ( $terms as $term ) {
                $options[ $term->term_id ] = $term->name;
            }
        }
        return $options;
    }
}
